///////////////////////////////////////////////////////////////
//  Copyright 2011 - 2025 John Maddock.
//  Copyright Christopher Kormanyos 2002 - 2025.
//  Distributed under the Boost
//  Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at https://www.boost.org/LICENSE_1_0.txt
//
// This work is based on an earlier work:
// "Algorithm 910: A Portable C++ Multiple-Precision System for Special-Function Calculations",
// in ACM TOMS, {VOL 37, ISSUE 4, (February 2011)} (C) ACM, 2011. http://doi.acm.org/10.1145/1916461.1916469

#ifdef _MSC_VER
#define _SCL_SECURE_NO_WARNINGS
#endif

#include <test.hpp>

#include <boost/detail/lightweight_test.hpp>

#include <array>

#if !defined(TEST_MPF_50) && !defined(TEST_MPF) && !defined(TEST_BACKEND) && !defined(TEST_MPZ) && !defined(TEST_CPP_DEC_FLOAT) && !defined(TEST_MPFR) && !defined(TEST_MPFR_50) && !defined(TEST_MPFI_50) && !defined(TEST_FLOAT128) && !defined(TEST_CPP_BIN_FLOAT) && !defined(TEST_CPP_DOUBLE_FLOAT)
#define TEST_MPF_50
//#  define TEST_MPF
#define TEST_BACKEND
#define TEST_CPP_DEC_FLOAT
#define TEST_MPFR_50
#define TEST_MPFI_50
#define TEST_CPP_BIN_FLOAT
#define TEST_CPP_DOUBLE_FLOAT

#ifdef _MSC_VER
#pragma message("CAUTION!!: No backend type specified so testing everything.... this will take some time!!")
#endif
#ifdef __GNUC__
#pragma warning "CAUTION!!: No backend type specified so testing everything.... this will take some time!!"
#endif

#endif

#include <test_traits.hpp> // Note: include this AFTER the test-backends are defined

#if defined(TEST_MPF_50)
#include <boost/multiprecision/gmp.hpp>
#endif
#ifdef TEST_MPFR_50
#include <boost/multiprecision/mpfr.hpp>
#endif
#ifdef TEST_MPFI_50
#include <boost/multiprecision/mpfi.hpp>
#endif
#ifdef TEST_BACKEND
#include <boost/multiprecision/concepts/mp_number_archetypes.hpp>
#endif
#ifdef TEST_CPP_DEC_FLOAT
#include <boost/multiprecision/cpp_dec_float.hpp>
#endif
#ifdef TEST_FLOAT128
#include <boost/multiprecision/float128.hpp>
#endif
#ifdef TEST_CPP_BIN_FLOAT
#include <boost/multiprecision/cpp_bin_float.hpp>
#endif
#ifdef TEST_CPP_DOUBLE_FLOAT
#include <boost/multiprecision/cpp_double_fp.hpp>
#endif

template <class T>
void test()
{
   std::cout << "Testing type " << typeid(T).name() << std::endl;
   static const std::array<const char*, 101> data =
       {{
           "-2.3025850929940456840179914546843642076011014886287729760333279009675726096773524802359972050895982983419677840422862486334095254650828067566662873690987816894829072083255546808437998948262331985283935053089653777326288461633662222876982198867465436674744042432743651550489343149393914796194044002221051017141748003688084012647080685567743216228355220114804663715659121373450747856947683463616792101806",
           "10.355073440919378699690477472036470469315794905372650461119520557405966250191554948388847794306923706128373152780234988638443954821327953543032075228882306893901540038156503265416950932549208174713640840670002321534598356631953006304821201484830902632165858733357596733427094375150860310219829690440417223992885533566388118595248523193740778052893647271700612136670114224280374211626787106078693643289",
           "11.741365414719872359995630593475834957852130639032528348910839621576752378207582766840832626901446337859064321328454096088656817987112409176281442608184950888327275070356290649885618053367245169005379545357345645235094776018156840041023694466493107510640907700621694994008944652785168884717808235919056023540605910031676977750466679901495978099025342154131430098841939052890393197699093697421676686108",
           "12.552295188839391064403078983058501313786961042419051509346988474726615643710688192053158471717695991904853941837387256340073310064341282022830248452137946273986542713154937759697687656073485311102053666502633006429446632794099471395918827441203897096147814416727830805903830888831170139661499031930347916453836121200110772307492771615813954882033814741470466415076136175748560866462690250372712624427",
           "13.12765917900902322346243585913156826327069465639834188399662474212182724479008698347907678965083909936781641397385483495362268256360763444317213586059490467781653518008717313901614800566992248144716374322429987750723867821114358979404591879648104388637035609754399283274014767545593886580941165664957384975221949645055255781874944374229417864853289321901220692566918330756199433299461690335766708584",
           "13.573946210017730027239177097297058651455670148127333215055495986565964988037166389618503523561570994165268599045375753512400565200059790703280760252213714757212213525833556727462647729566258301490115414864748436290794191111843775262434063842232241185107471214859854864335150887423144285468856056851239653315249148749701603523173627966758652109489308872414194066451074827802642589890222384815709071028",
           "13.938589284701101831675037903832444894846690407420628610481163937380536465305805316181768586604603341197033655576371082304146219991693253987043452241307224478718013872459287813264283001906622959681872321576087143975972033651747378310838673897181549722528073649569290162541997276292708638316653324605571398691171391912961431088832002969546541039753774950645926800096854799942237644831671997679785482847",
           "14.246890620897408672445451566098178524763429935510610216690355229695078778430708804421870639655370744549370273426124295240527401708361100923871708589722651960428863955835558634640426846383161118062310502176054546468600516588908689567134867002953229862887953032595529925511011383644740837050282379317993152702704798000312156694502691021267028423979454580574052732764336948692001989411724963233511159783",
           "14.51395339092117324610526432755496036199337959182144260442491096061930765841595850793460014941610881052330595683873016973224744403781498772061971443926277312865243350174522677227134957935047605429941424114476598970997223505479765426888886686275495180653025817201214478249415383537946915840293549927935267010481189550187682053892807264533011114305883442850485847010726074555176479027986850883182954118",
           "14.749519451795537716462643672888549370666057639128212044001868590062813335472483708120056955187220713486931180221186191741860718823347399129280491176499608998857949407351247432513825095161942372289203657231256344909997480732110798704027258108845584594081552703009126596026870753341888414903979390651641367952487019992866762344920748102417282166652161104367292712971759633790964534907923779335451109079",
           "14.960240475644662390024535772058360495972689995692264619118050620984863397335883147741001175137837088856196207660064061839685086120444324885420172384357780225067863432546453923434326358834374021700709153791633034800554630092122384413882780201869241166113270104158152592921242256738530713688528178980864650875690860545577283508963373084295388766350046816560200649161822618828666606722223620871032632611",
           "15.150860829728925816270269964092673433542841210964830162793225770535551518840933315190406262273098573795238063523955485390896001670839635976377071048140363687808579387495186737492639543519534774851635101169579614712607935494886581440494071142193798729857001548284675533558490478467589884332536443679208006567281631805562767928329937083844501800478013385969649342207304490375728237042482944784544838254",
           "15.324883579506436493570233597448883519693351814374549397498489035148221882270042436654099306811313261346823941672888447313628931256927773946985727814886420570056552118783468127543676769633344136209500636666830026098080195180498059221494203817961212365532443050015541947762494604799890429565189588245725920948145579035449699519788465993549043152027361890729572024195346130030456992216388648768722394836",
           "15.484968991583560712862164152602532972505596882865402498065644250737301955125596006515274461619204302558285733320299957799185995441209009451408808376560872924206710117585718578727591708553957297729203457443101522435265312291388453827953066056871634785359329503731609907767433693085428240202927311226115616775436485821324503901092891486358733365244513748729928672060637178808283856096451123227237766837",
           "15.633184933296400322886411216914351313442169047575854720677392808712745585778888867637029115450088924031428350724038141381219620588274493330817678181565157141965018413952386880843987389414299029950277357119925470077396582940505069596480801954154029286187951921181728705116872415271634077604874063371227144842898712603028462623610688275714711124501733417572553751110494225479183571648948845731888607993",
           "15.771170674177108994750017157264759379388540187789301748807798065200414118407271774235601089093934343495795521812477515931432709195503245641975701807702026215615357450518151317247119115864533992757135781852491298348627270712291588580073932563653614422839620206266586292520948221873093053975823776289996754936187304352159896156876771015468496811587576227636095095473566772184717633945339405330371627545",
           "15.900247714739125237305645451417508966665591345786220672552302840334525160137723176098165167253830541655084109190400464710649206788771240950251243759258268060695212680017248831015574286848979270728739089100870231114250075432989803029255029492399141798408644860336753530942734586654467728627658977642688053752033223720047083446705808444218983029996407752310091763448795485193086044050699179075227514085",
           "16.021496956952197922992283502641894579962626483154818954043789565712934054541190510263229676242727892630672492409948339732534439120465067507199758379167701667321617794907607307728874859663168072318224068053703928672394393376652024298292046951118253912008558176064907492479298408285769067404165601995001682495873893492451380569168665052480684672167775365328449822100602627386235888117607629689989295358",
           "16.135813783442291449575041100809653925479234724736343542177079039059170241908162244448865517513646317913575323823536275208044355623776985732503939829900060864487127004364547143183319169525031486430688761890355067541570726092642869160866842639424318480923069734468346837396224375681293323049896437786794370196393419740684520457519876260717581058279975484300455393801884203407282602155622048760192869731",
           "16.243948224975912162101874242338089698927747514172297963501304802721456885123447742525917146000081948350317670558217990170315910524955950657647628011646020228669505156094231753926456488583104510177838897523717492728937537629972877557426292386450904986250829454391823373783104867874948543639133365724217397301073161382317429508046689262594057712721480553629716238053162259132557499326959178271755311819",
           "16.346534812891312020017737390565972522861639520956794183631801030677318559517349930264816297286373068267542772472219372292428845683416042271115312027071601889325060155482892443418582095080544516697369324049179348198348320350032057951783059238791085449471373733253488008998200818236407442435705194113745206455080467406424910471012832247972967952311217374959559350158619575075500673005632848486334892081",
           "16.444115140490340128500589643685615592593325439192231430163649023292793816677629077048173396104592590442505550803932707806677495894373323443853064668488945973357469342360809555842080876327526869093152265593359032627507962690981132695345232937664935403696772586286371082469152648334224734697834132478315542287352614722743239776066558016553826776123247656549922477370984130115175375533899599151832866777",
           "16.537155171118865136534989063582757364253199496897225305411818162632163606557745471767961160436879104964882723479325928591604518246962419687548390063992529356693298596378658322800277364077966826234091623459656551097008920549618056923279473982070326894236928331513682937597704431571923632859325944958271156630648510192786810276479282782233678225160061893949554276261365074478039201200141096034185340788",
           "16.626058695701082390621404187251773798561764113151150041596109882112467488919141098521165822165293765039118029510544981683905860141438867442019067534265891760023149100317968826679154566584457042877767867250357368225093921585968539417839705388349710086788853906351301543791493808667435475597486845630653110450254081800953162013336880338884388603160347478448669890791767227275369657780841518987171046251",
           "16.711177924047687436041681081297469757517395732698719962499711839818824068470368594816509809165130772248099176990649818696451892735033114399817499272685517472929769530848678764490613732856699010061350335113220658748043901782349324603111802594973838523650919828685995473213566914734286172266839140529583990377667556754261878256224596588519392559788652514828861263680954096492024398529162625680677258884",
           "16.792821912654942574589969785414406627855177803111176492774370259331640542125643510644899208865670120639893723574191986313708664152776559897034386961403215067476041927571306948209653637016541256814370701598681758297129259567548154765775910932763810160923332344469835976310776610418278334876476195529920021337399751846302536974621623453741347064741080510495703994896565552519309503521053051550623483165",
           "16.871263338577273117049933407315211807184894736082799946337588399915643695388836578819581481279897851015095335198849942961931601540668917787952832251688896660707943869197097893837964754891796695592066084995854170120209827918603483811654385230282025426754327077293929504883381435192439219366582495078959939830875435944938000044089717015334115741229281828493468876793032191535756728196375330663180727281",
           "16.946743994200631806033476403925371852051109645526740171061759524387799979304956484613870531865187139791227191840781754383935422487885452369507952721547705027535773654885832127821314459521240798742974634464887793874048016231828535132695506414037947185201173465247385982934992490469404967587724592554548395276895353595409898385619478502390678687463962622836411208115179925881515699871702165077675857036",
           "17.019479282236065828727803356580529388797042464710975507911282243104151862598751845439937481303609237769338953879291798328663979939210753132559510492288100384177867237190989151135211416672909332732482371326996189543048703303285313799031214641216166381411885795576234361808712082618702464622338883787915216150436372783047860816587095332458688921705641277515576461385290501276378977328383137813013202605",
           "17.089661921583428533920885090473894674921519340994685086552627218992626938319872359613748323826158856027837200411438668460410538403836237956276065714403575420148736981655027235218177614075251822388622713232585428029969533906061654341819894513492491624632974192525592636008127939368987600981961744622054023981036109865947530952316809705151315101400446930619393266624895135955182323721805890693892489067",
           "17.157465024686670167940702199808038440453826682770881229464136716214305217066325160993039686330328188786987064692354236198505670650108871009195717920051441546822338399371747349503874711833721760676087218038457960626001087951737722106836691584985009357268276351001658789309901363541541142239028890280836555429745898638761562947010206765922958485242984359891337645169887782296197402533921841201821435322",
           "17.223044670108153088406971612680579462250390145455205711423224535559905829397735171751863831903325186673310251936245608147861310604382786127429364616442074955520795624793164764110780962566943990430036624945190166480864873260501647262851147587875798742974124100786387378932173276365815649310057053480347330487890156980245097594505867694166591312545929965230291418670445516669557156689966248314016354044",
           "17.286542066533530981819924644041556546839394128235306235416269038138274222049809140847092268150839206354707638710275634022335650694777681817694798501418812448888688173373947001904190184008892957674606496333161081431327984331509197841574297574788702287443502123119313877415278877304866456565176360773848651195816021879148876985173267277031263646152871873014441183590471141901167533828621963866651938359",
           "17.348085383681499172852340212837946913681349503319833895379599364916346996678774115472356552172467728498192423708290186696353892633802275272186862935888864564436104200250553583802585096479988444811879111645109656880728153108808638274341978968354895482288870990505393394330150686285205409461910256828313237800771316014620729111128197408712519655232673726993167731090657784157432284844410962540306871812",
           "17.407791309811451413991402934776587727274692304168979758129844486195147501475844430031581045491342304862802756719012802717759227086944920237646742600183730387935914237410099970497154963215669635477250725760034883360834134873909027143784365748082991457136260982918509631214089379283854791699127985575276751950639814381413022404340898037317031344649130788109442934133748721229332612763230675387867060167",
           "17.465766383402858316731575043043871754932551869660069657366289253587674381952187793692876332186097637634087898052217367730831337525450143144735163993701984235559102178727289467801811444923511321849112339058962832869205150513708620081533545907748750224381419445198858131583714181742840318906599828122743306582674187117835956409356596804579817623463035229848008072471150460106195548408352347746742847561",
           "17.522108137193897711249955720342223562404817078270264273905715197834637314555104238731504974899534359198079882854402652102198388084110484760619705504323190957788898675675472299192322756141627162323541713415099172067523105704142354142286507565791349450889277448545966302507723270185283899864397011417078910276215036241179611950839739662625626455044878609357243901835302874736030320059872773022839076998",
           "17.576906085439158795399613853863097303860835367703382634217979275105020336999327656009595436350031662565840865227976662242781789960872250051998481804897622287887086985238091438506085053631982050247194140738539987379268409303934286420729479452837709138518786355198810012076646196653084861659638299340083440855659993917810136693887499673110592919880225890667600561450738971059427382635508450091570097859",
           "17.630242579482716539323837128846626753956194314343736063283695118187974605321998609676585457221722093540967455603842712855190915316171267768551204188316976169082878976713161041600921069589708836776259260899700342542761959107817464710887483018031112156977258240064220687983570300417350906046032596793701224117336432612122346984445322343505811317200044404325245524587348610225714892960184376194438742903",
           "17.682193552177642930469311075041540632454794315173833851548924954045201903733019976720709429894183669091622069055829864963447674638381460850277690319942206492046322144914184638283615681017449686491773099947404077379395563267125512703092895031616346899622091168324459335139948282486097513426616067084251033517363717148785943762510534333490466778398817865864252996839848563898275763467374270388303474531",
           "17.732829168042892302590066897013224260526349617442757064898532827287653373154374250085440081248759495538523122114463120786358068331952209807684385821625688206899889794184992102701935895298679767555036775735985597715341448247873700759325978422207409167388561946150223833135803435044691082980471455256218231653367433009869225710546713847446382266908465706428817510556416763430212167661925983678317689559",
           "17.782214393127773091907390576928321922818609345162469668288248606737448584265787291899115365824350981071781548351963523796289812496283393676088371363622146243894946956855177324929267339605228930435844698103907214217720043808120931206339233299438914243909015437719067832873971853535761149326746352476616918607787268605165583917095714007024394600276277913067861955743579281638075879563801550521361400101",
           "17.830409496196797333256263738087407756305793204876426547866274663003566432384680240879632999625725708300091127739268764655234592101254713785942646078437589939733647676861210608435791914827456291767449764007558390614584486670558546409061408823289149666666818024559557140308187730981247373784664587280435856629009596287509432264978977660915985852234235063815941899607957940727152099219426925372856581083",
           "17.877470490934232288852811743259662703350866532641876235327240424382634438025700311951299570329496669487738366928514133455422447732866797861412283887178678146519925852345567722230587799483201967913067391020608007825990349366451164476805546339769760220805534987675878347990806759564468339408725207557557011408435539499579334714364702076041798486860978860009676392441269681660376828431489832998763783258",
           "17.923449527306267869903426070004616586122783729793624019886904269965872168854615384174781325332961537402467089754136043237824345234828497017675462013908498436584882632257634259557543282397560432131425953734774375224901224027834174376390115551172850840545446370919691863863162138436779349056977625063368614038926314141157758995633570593909444840439875814859101692914284555504462737973961801589348562421",
           "17.968395238938123096275003449517382823058653367901994410453731720353712579214128295617333869516953666517153266184450063833611730223713367177888956257642697863674682557764474491249664869634869467450632518845976225157445436929203774807794710993549169067538180609230496568688211520386012533845072853251411123534511690847284034061828924979836346301147669327210040586682352268581578885354613616677879192008",
           "18.012353052308072938755867264876761138384940766536728123219303502872235608220456049765985905816906980541771912193578395286636524639567005562096644259687421270993689984646487799760867557580378976263785987858028882916593402367582926712749625025395229096162602944095765767003293442756802839571379406129666952518296572179431801499117577022831811590456125422748646995266409456422225586206021840322692382857",
           "18.055365462686668331727160748231009519565746647193730385530909331597852334952578425145279427250887034166658709337134491235522678908221793937220809954557808901763389178392831492051282059480629151432246720826269576988663211852119754955967569929315382922357289706041278636253185833298598511399638105081736687620508948263244525471222947804257016135464845038301780535378180021273347494295813215232102999649",
           "18.097472281022918092839876166164329437973263728374666387409900363304423095913608188979786368122872685324785740303523457754035383735336598870745063200445496969465305638435393742921093080855121166597064087151732410717510776813994536867846471600876935467557239720329955864393503091522335217308468551898804050931884864074974187930869095426636541733964836470414482229854405996155347764481217202728996408614",
           "18.138710855372574931023279730094645378686287349356676919424498204049762635832652703081045863264164065707461503608812437952726945167389204858679677978907656542126675434222486013764292798698974541262372833651362936600851261171024940667498778746253799866490414715539561648775214507198558445051277115507846030240001071535504590488750762226492938922487331198045407471864288427126708589240674067562104543957",
           "18.179116269955114571377487963612019905652085216126618591531831326851631273535837433851652060806307371522002942828348086254592323263632131183438294055667010855067061587674644272103122258118876651912890486407250279668447142712674562099014392095571206458285314302909409592642841502932705416119595372330139185232535836986175505626716905102551635720868328845059106716603941432470493828723765413424721976223",
           "18.218721524498032554193215680552580137491683034589488437319060302272592206508827557611431375182704552154497537575444766904464774327453349109721410510507102426285778233008253433639863695380913463970388673098705619991229471182831382051550803100625573309055088036785775202381313835883262855970305798706681563291905495590616431781211404743717763365778127660253455212146746026811363929209719531660648504082",
           "18.25755769616561915104320634671335949819923250838247367135480870103002754055972575277617781066800125225275977232426410882766773084485826986957597205108505829168630501177398318211279559799848433615486396320080576991739828304477785924273653352846964138446264644904849824507615174857436913763769835408524950658917536946876135419383532210821743200063786350760207420579200676223678007228393958258720083932",
           "18.295654086063005239961930900603578763176327716162022442528659338478895027684612801134718048445578078864512023650622689532231640803798279071247213212940626166721703242773412064293396910008891332614727995930235839898502680907757808722725076622433926254807390507349193089487065273859422711581690536923283818694448490391002663699373577479984089586448960986922914070154319198856799480586814831572660049059",
           "18.333038352045729366070360053860247416881993539906325936417001019639836810235521744851872438194534698737793818113182566661284551265375926400252096198917463166081789346386579278436021220544269108158036360716265068681920333271695665204550361645092770414267870574761534021576748772414710332296686656677922105572480136623006540318287771752403257225484193416375374630295739694584359181669584822144217946089",
           "18.369736629342788833545154829965612197972030184654204521583836723612431388284128470120361520079152081604731214550025504583810749093225662462708297813392905679444268263153178673045591535099590243771860299493196201337328463166245762316637806913270164277457975112426005180292583426915918675383435395598215385153247994877265654873384456838948090969392401887506081626510459117637684719849016393221041720966",
           "18.405773640310900146133080088261946202976690317466294344672811637024889157611521518848738302063682147953709444704912322724498748005402509152494514057803322060491136048741666988817114427555111210963923209356023892097003244960804568051496345885676927757469123572277620624446804627512970828422692616021227425852662243212443997074981005547833762050079432581680067106949297497692327125068990077901522226957",
           "18.441172794474399698182030074721740031645602552534420544604120821167467838337449892992771747357276011553142091110875268371883702350672621604391781630952792517951394285585851496086739721764180588422789463043949353454011941863375479066641639870318143091000764755109263681792932398864353035859720611512874738125344871434992639512151401272538083058455878170262156351968871089395179837425273523581266914745",
           "18.47595627986464597709414934346663732189241252367556916529377819718435871713823800089497791390570676646849082185155969226893554048129639509679051447135722658187695404706235925157146425104283059827944524324932672824661552843990966903508121362067629390223781365207266998003720499319458536728358620338853555386727837290179454339519141841532839938596761951045619471589862830891354308681626977577420483375",
           "18.510145146551442627414905806735913567003712650812295077443330836708330498077617580007887630002806998527489958357928811323358164284670780912045286972377274734553167811143903532773133805470585887337986897779623231393980154689146471547983633489101760043708991315746940120025902341173853793429617026275691215584867628750464940410772709161351434979639911956946479201132978330304672548030931942250342827294",
           "18.54375938315397840777373858913955248239941847920328667581513228848814216209238117867823746808674105485439472281080229047484288277191323337476914908731578203036561093352990739771860442165984753982600021729206737097284114048195126812337180690412756158211962923840645830361425158816448002373708570542607328640280411811505735178290827333249222248570577590327817428193141575383830773736064963674654660788",
           "18.576817987027647186937851258336640937668818840595786321214115700990580993937301728456982569596955182509352640277092493549853424411966677818012248002654715239100874590037738307469713048325570874520232570867464224788639656453937494219294174471955373301736672685268384577575584576576899158917424074434177408506824723823546414342198182263466224674661649356568188397290822447447185825387180657594093708751",
           "18.609339028743835443520046961477709286939310853034399826433866146117641838882532805452792636721283076369348347017617290496488517658040508081002292362699460963188796636919359563770588812590411785597112719721757265655201721601867128800954165396591923271771796713301912904158675092529178443570927414653982088273331584295588174225100793199329706639345162173054259175639620324704962515101988919574683797276",
           "18.641339711410638424722499047013449967887089663871032284600678326435618587023360889502435665997102476809993062067289899429492640057255987173558009597289702007868879986681400023832856533473923548707096670227832737339287908558274044368146372966754230837592655179154123960009424338853507410779228164259856997555111697864450713171508666161000757176178710802248263025585993364498172063037657704404452535503",
           "18.672836425322050368486113163967894536152542145032787472779137579977190277914128508177973702344749215422866489255932015284354044054928123475206880037062819389112504714106095639732288437854189846017324055713142277768438434138800721340366655917386856186887417022606078557020556496094324427190459240576788558291551699271849591355480496655647717244614207638067307680084952591605729876523430439516165337322",
           "18.703844798370253311556748128890196309240890383998201893987314003630502067675284072677315790931434159739316058047067634433988116165753705089479010402482501330164866808933095532965233351834719435836698735379881853925522288987702030512098787246242665060494288818182785084700362957305908534097737840396555283267034806128689898134337131513408714563437722088348519474128730338455257510535561818243515295123",
           "18.734379742609172948695846511449581634565683544125106511957075355113091757584835040550323433523983103087395353257534872811737536865777209110758107808331702363999663358869296693765232701087487872606287651215678117982769333263424530048349886338191672000649776344735688777038908317935895545479771750928675097454702913124841960320021274672197296187088995346243239131294305233657221950357770047779041168948",
           "18.764455497316603659697964676682244441947566581055018250974850982088974686459881812049067093753928158863642280317022246479491132048354932411165733919156970301233277913734990931824860100667882748928084034147822613522025142522120240087012670818451468498896073922908905983528384054249374818979283244638542138214163225714979901542761366770616662189716515808719831023436094002277345660650710470976214503432",
           "18.794085668866182740204823781307373749785233327862921970091261536522099678420128101122831261188800902247301052497120590553831148856384993334702795779266689462427228385943774192181671843132138022659399715350825905060669445319905454600530058604054055511420322690455781203696250448217551628254838303648547201462249630753150471851402187751539728462449395082445265630322991793425964521071134426892761087686",
           "18.82328326768867946516209716586995768024335839807194833176766112965627616145648993732694510000805245875257813926012495009414082293067923695618654400188454857448056930551075037023425200611189368317293241290052197494133870263300149851324237139757643613913686228545143709137199633423670636305032456696459165988858650794045361585281861601603030153235699981659298052458840307661288883784791255308752353561",
           "18.852060742573912900871064645288864648159675216435305629278797553184221367993840166607573470069267053513735624040087819326415506386854437946403720812228284865900153407303755694944652909440890918000505344963394853809579466102995490946923404655914747241389209313457700354163158136320973199396402856713949800424774426208733366034958384913000571799639761622099679737940788622933241370717844146251711932128",
           "18.880430012539655701065719740012642650514689061888847292472446317260576484360479108374318422256748212928421929786176000015552730066213309405093412502424231235041703778572229968621812094224953783751727048234887878221529796434400738059144871066371011576734445290684672478932371976691435646196856591582073749970279840730554499319160664075936135140351358832332481792115613668240634181554442075595951874232",
           "18.908402496471717232311467781165733233171140941896760455553034330908971709296619686804159498334964006649119069488646486948675866965781697617293443993292894960273992948681816600748607581132058645469709895004815813032481899164996133398853203604094641445335770040577643758444125343960451089873183667195183097301272165939070294009329717796749663916249651314647264543061678516229808974080105230884519025421",
           "18.935989140719681524355183827184271883117018713616424044375161718824065979531438908152091359516789236304421957752893530968312896089478807820652680723918795442056690234785900341761515097938048245785367391810983836804772037480360220145546699221066508593371664219887047644628393373836620900288539062774663891696384999640649150431067240627749142322666676970274474792135970231868616050929898319819448939034",
           "18.963200444815204167404888798469665227431788358892299847438551435971105813566329353339851076750530262901184430318825907802984559086828099838800823075902608236275943927671387049630228757166718632225359114181378374769186216486342164672374303171089906235756037703334480067642792992542669516647312965086899057259968955796081732571906843773036664256360263051447703790357359429377002031207739864979999279347",
           "18.990046485464088033353325238570094871523592746407402454864447659438418650897352188003827603659585905257296788261872980663015351591030286798753545075312871886708540373690732957230479307213846083134185408782477823043603415914158563765104431367119681516469558190435321626296408405773925785846797696565762444462069369032104851518168840362055319574446706630124442642806832354006488607879109906061011225994",
           "19.016536938949335590921908878972071081608578803308502271146163132935001206102423418117835863551157682291011036313549071538079474102791095379250735204641651869626255407971202952320707788423726057706822543117840676308428340234652209177652929619836857962138281947043694787969753375808877555756525156889419987622193160096718082756877689454214937644538436622120458515758071395625444960838247998451767464267",
           "19.042681102069820040323211057777714109150558076228575985448688497208727928412122242888431932498284067180966108960090928128416538326586163291672353531766027111567123306015724614143020205136706498507055346813817986209642505418179828757157587146552894936306186818734849750397304488535715134906592464324381660408650130874094490704788761416060096501351661608424222204094369103147551037672603814369634649173",
           "19.068487911727958175697030382906549671323284692477975980792892280404995909757933182996822232452584791269433004172376138676679805214029937405307084300414313959313789831600985178255638111559270519906326622715617717964168412002699242926437873012736620082535900009120622005729068876661407020205549742970102634687260159098231611555026386636952356914915597529253373429394903680674472796958663494038067669179",
           "19.093965963269656159901801532857652686731313095114787279282350143184060615248880952958892345276101741792959203039945279273255821413683057071718873479055914318955623580559627285483449590741119271240498698177664589148961791164562328382134513604070535948997556425774535241433839196056647337630745324677877037589790564832932809741759428622931327986353687891898499439791429917840804474313864245639140963876",
           "19.119123527670705331793258647026174473061813878914781070641605073424527485626683566355878191198143709838509389044950724156590593262994411423265514055870555330957166471517671117689943118929441254707396532385048466111853801352620946858612646592880180562600210076019427990165200479741063391057955342589569972650353707736819649707138135073772494722116787681314798945037658461538374603477221825614061123586",
           "19.143968567655614969812922707872905228504499895574401195919290684664632292543319151929670686365568982588326143748358606442800755409560708661213139514248609009613770051771501624762852458893105259920477327438788243371302722213272226600736522283567127277667156871770413468147870394343816746792945999572244176333552811712447483231232773171106511438260175071145300046115843401710072061507331123043607901766",
           "19.168508752827482780609154136923082745746771414212427034681109262302000645968087085215520638009496537393752178754873475983837687377025695010556483691979309217953419255985001727184619660158435191823702770674175581797631578786818852604111682506864006199186805540809913091091561814668081471402749434233431033452441548505582706019452955567976154397574205270457372773342241946969622998755580434262211071558",
           "19.192751473880834042563713562225153433521577940186973776803866639536641104273721851582869330116346851241224039623128541122821371810756925681002641212419779923811384320279466454221232217629041792882431116499214977303106306157966687858071257893713340199820635963233697984370035461108601863346164084264967404520613456100656965925875472210769471793070984755357524002396981276686789870757739219539841324669",
           "19.216703855963329593992083348877341510356872855419154524286060224198192164950344042194924570763041352943159776496534086655621090662648765929073480563926604513272233491262353914647091955902924297702861031248271549558718609763974479507490837350364617085882753391176792131259310397149374028556893735115741447201974608383790314541221313482249282941830681505989160209022143487123591182161037084387107577471",
           "19.240372771246783136336288622974117018573581443774444767997147210768023102387819846722491865244820295698055584478151667485670300985795814036287612086238394489310685917822308982062485897587424683040075624404469719368214374255580054164790522993745386564072906888203344350441788631572056395846260169628655731587443716443908470481602481327796867951175756539602824534024677208143458306390998693538144276002",
           "19.263764850762979509196317705761686108092922892356892210254799469999751475179529646750512177469718303426094538082717842694993912366018532429385996195100165328675330327321101942399890523238950583083919577089467043577581159099371618262297126999910295820866611740582812686480215151994304295620401944519281853807987442610790122820927006439539980791908595120476521299636624843377591628005484740938263373595",
           "19.286886495555294522450851525927489255646333848651043717882790718796769943763203547772413744395083720600670009266257171342692898145198064660758379840502214228374875168690401725272168814130387964227940171355345107459707413634435078254467165459318134413588306046635342334572072865760417877208230914828250259039270224031775041480686581149991980042698893384296012208438178584526032640636530664689171174514",
           "19.309743887193036513569909003754932688390195189494409126081957694707453295769725463252527373922995550848707616892717734716346726926003376949988744940329952938435722400261306789658518736487435455592323288866363564460056999876237001095092480820045715217327131479089102764865100373961878408215166685028270522348153253852064527393390794401890890932406298166622224431486187051653736399833197687524233000392",
           "19.332342997691718269365647851051829244300572144237799985690236148794296859647334840581772744190077146167636205171422529047782401411481872480699008041273574144705390909628455554205882856537701103044535844002047328262877230706418721410147702147287802942241134713881576962004492246225067150128745210304021219616585555420191917418786300056829952680702638022710700076784220324428140020802898115083347702904",
           "19.354689598879088211883278917315015191222443867565368209411861608347893184500809153071123456883339348988675265028178228103174790923741638364554293709103213239495123865898742141584076941660965692453426480770865774436380303703020587851939728834028918714275983773159730132531253761589283704592841895713231304417993106469276962707965968679020792759185611101812841427933459395519877074670185732532077876915",
           "19.376789271243668798743209719830951710660724013826759987563467708214368337563307069810449676142525612003499201086112645501159094078378318648014400550345870757023544381773094826620272453238529286555419097429861755219673554476395943490245768513403194721941361801918811127988057063915654267554707302094460700078255016831241731287084620030531370389406858605301314913503432729894455360413554796057743624785",
           "19.398647412299738536733101983020344760416543000933452586938357216391679714568872063480206587627849607725892049860100424409512218327903976557192129920072602449053963631370720893361294311430877864798057636106390383386608343478253049758287190639090148001663846660919016228401517981946486814580501445191763231914170223651871543954569563229752867364010807484149328122287932691728836607206296504503654291916",
           "19.420269244500125639443026886412358511499173347101740173789245348545649777799992933455717128782979425999232096035698012165274922488616903998818326535687811364645463339885951676665882203155018792792505473796600843314336576346118972683035247786115152970665197263535031125663311669153219106967873840911780618890220192918885584791820817380351857621444438732170662912318105800373867459654881916254806601924",
           "19.441659822725832764853886827062860938003360575287073851586808019473777348197590834815623551815029267586554639358825375063360984210103069003729612683914916394217674289812465439813541316938870999354861402160375582810342455043995730819816948505547453029096197833928688735935982538915244747599958654970407704104602569366784225221531104839626781845870092648147888203880688567605208831444945817857785153427",
           "19.462824041379362514202453738329145059871795167945778128620896576209292571177865023083134487492542931078517939464890630420523665550799939651548618906955780515811541533057671943504991862669514142328738958049297858093270685582461594213632344682911288013670395835728596839923778216755784200501989248577166652556656745561631320914185098036550279972269808563352717494278133064808018424486494613581670904697",
           "19.483766641106643718481178395114340475919752681994355643610181441860832131581914604665304968112958314746911243491535376957183544987386808635080341126118126364286470448001964590403332961518317007864293940732767763315935539659300785943459751426720740528037173390794501709108627539732588206226415399674048608843262175783293190807110169521287571898125826758063922707869247389042070537305271065057452360137",
           "19.504492215170652185153425611551614568097792837031617515988255748353971463776213613651972980845739009653411602074763823583486335276886222365741099356087975815479321535979397651959714956358141934399110351847826398926624666313434326948757230737160647513320407575743953348133831681915668076630219831345596329489041495050445409516995558844346940248424949708046529076568535423815692145534030368195098701895",
           "19.52500521549816144973266072401605164852396136071234571245810367391072812166019661923143396972741343266046339927362413412251126099193801965681827360699083448218860273549526903661234475357554150017778848244185893299525191263899191588797634058235164494448438674653809951435022502560679429105527325418829543863756825670476364629140951247734867887985343458291386040322597855003211338820555010939867273248",
           "19.545309958419535630089123088329455125777373731767323301911891787605488456921558805681207304272616222501167615837357366396904683096528060072684210523189673108467018417135186800678628444987751206147451054525963839562634340820804697604369657105969423354842149829173951943549166420900709500398322624130365195674097282456714104812933002700523506427441210494461469512180706693823193924457440917371176532034",
           "19.565410630120075532420489068555321524954362381908007338302983473268528473151341321768804731192210840056455228388636731122203200610869021021871133154865620432257415783302175345843898977949742849569372169004904667621738405749986270376895267386643049526766451459390277406632576412790748533372863644721976845168460840162066672290292749672668879524315410824251032084115058493720402881895485689493165441193",
       }};

   T pi    = static_cast<T>("3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582231725359408128481117450284102701938521105559644622948954930381964428810975665933446128475648233786783165271201909145648566923460348610454326648213393607260249141273724587006606315588174881520920962829254091715364367892590360011330530548820466521384146951941511609");
   T tenth = 1;
   tenth /= 10;
   T logten = log(T(10));

   unsigned max_err = 0;
   for (unsigned k = 0; k < data.size(); k++)
   {
      T        val = boost::multiprecision::log(tenth + pi * (100 * k) * (100 * k));
      T        e   = relative_error(val, T(data[k]));
      unsigned err = e.template convert_to<unsigned>();
      if (err > max_err)
         max_err = err;

      val = boost::multiprecision::log(1 / (tenth + pi * (100 * k) * (100 * k)));
      e   = relative_error(val, T(-T(data[k])));
      err = e.template convert_to<unsigned>();
      if (err > max_err)
      {
         max_err = err;
      }

      val = boost::multiprecision::log10(tenth + pi * (100 * k) * (100 * k));
      e   = relative_error(val, T(T(data[k]) / logten));
      err = e.template convert_to<unsigned>();
      if (err > max_err)
         max_err = err;

      val = boost::multiprecision::log10(1 / (tenth + pi * (100 * k) * (100 * k)));
      e   = relative_error(val, T(-T(data[k]) / logten));
      err = e.template convert_to<unsigned>();
      if (err > max_err)
         max_err = err;
   }
   std::cout << "Max error was: " << max_err << std::endl;
#if defined(BOOST_INTEL) && defined(TEST_FLOAT128)
   BOOST_TEST(max_err < 20);
#else
   BOOST_TEST(max_err < 10);
#endif
   BOOST_TEST(log(T(1)) == 0);
   //
   // Some tricky special cases:
   //
   BOOST_CHECK((boost::math::isfinite)(log((std::numeric_limits<T>::max)())));
   BOOST_CHECK((boost::math::isfinite)(log((std::numeric_limits<T>::min)())));
   BOOST_CHECK((boost::math::isfinite)(log10((std::numeric_limits<T>::max)())));
   BOOST_CHECK((boost::math::isfinite)(log10((std::numeric_limits<T>::min)())));
}

int main()
{
#ifdef TEST_BACKEND
   test<boost::multiprecision::number<boost::multiprecision::concepts::number_backend_float_architype> >();
#endif
#ifdef TEST_MPF_50
   test<boost::multiprecision::mpf_float_50>();
   test<boost::multiprecision::mpf_float_100>();
#endif
#ifdef TEST_MPFR_50
   test<boost::multiprecision::mpfr_float_50>();
   test<boost::multiprecision::mpfr_float_100>();
#endif
#ifdef TEST_MPFI_50
   test<boost::multiprecision::mpfi_float_50>();
   test<boost::multiprecision::mpfi_float_100>();
#endif
#ifdef TEST_CPP_DEC_FLOAT
   test<boost::multiprecision::cpp_dec_float_50>();
   test<boost::multiprecision::cpp_dec_float_100>();
#ifndef SLOW_COMPLER
   // Some "peculiar" digit counts which stress our code:
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<65> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<64> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<63> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<62> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<61, long long> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<60, long long> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<59, long long, std::allocator<char> > > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<58, long long, std::allocator<char> > > >();
   // Check low multiprecision digit counts.
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<9> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<18> > >();
#endif
#endif
#ifdef TEST_FLOAT128
   test<boost::multiprecision::float128>();
#endif
#ifdef TEST_CPP_BIN_FLOAT
   test<boost::multiprecision::cpp_bin_float_50>();
   test<boost::multiprecision::number<boost::multiprecision::cpp_bin_float<35, boost::multiprecision::digit_base_10, std::allocator<char>, long long> > >();
#endif
#ifdef TEST_CPP_DOUBLE_FLOAT
   test<boost::multiprecision::cpp_double_float>();
   test<boost::multiprecision::cpp_double_double>();
   test<boost::multiprecision::cpp_double_long_double>();
   #if defined(BOOST_MP_CPP_DOUBLE_FP_HAS_FLOAT128)
   test<boost::multiprecision::cpp_double_float128>();
   #endif
#endif
   return boost::report_errors();
}
